/** @file   objectmessage.h
 * @brief   Declaration of ObjectMessage - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "gameobject.h"

#ifndef H_WWW_OBJECTMESSAGE_H
#define H_WWW_OBJECTMESSAGE_H

namespace WeWantWar {

// Just let the compiler know about this class.
class GameObject;

/** @class  ObjectMessage
 * @brief   Represents the message the gameobjects can send to each other.
 * @author  Tomi Lamminsaari
 *
 */
class ObjectMessage
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** Datatype for declaring the message types. */
  typedef int MessageID;
  
  /** A message-id for Target-Change message. The parameter-table has
   * @c message_data - named parameter that tells the id-code of the new
   * target.
   */
  static const MessageID KChangeTarget = 1;
  
  /** A message-id for message that has no predefined meaning. */
  static const MessageID KDefault = 2;
  
  /** A message-id for response message. */
  static const MessageID KResponse = 3;
  
  /** A message-id for Controller-State-Change - message. The parametertable
   * has @c message_data - named parameter that defines the new controller-
   * state.
   */
  static const MessageID KControllerState = 4;
  
  /** A message-id for Weapon-Change message. The parametertable has
   * @c message_data - named parameter that defines the id of the weapon
   * the object should use now on.
   */
  static const MessageID KWeaponChange = 5;
  
  
  
  
  /** A message-id for
  /** Creates a response message to the given message
   * @param     rMessage          The original message
   * @return    A message that is a response to the given message
   */
  static ObjectMessage createResponse( const ObjectMessage& rMessage );

  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	ObjectMessage();
	
	/** Constructor.
   * @param     messID            The id-code of this message.
   */
  ObjectMessage( MessageID messID );


	/** Destructor
   */
	virtual ~ObjectMessage();

	/** Copy constructor.
   * @param     rObjMess          Reference to another ObjectMessage
   */
  ObjectMessage( const ObjectMessage& rObjMess );

	/** Assignment operator
   * @param     rObjMess          Reference to another ObjectMessage
   * @return    Reference to us.
   */
  ObjectMessage& operator = ( const ObjectMessage& rObjMess );



  ///
  /// Methods
  /// =======



  ///
  /// Getter methods
  /// ==============



  ///
  /// Members
  /// =======

  /** The message id-code. Different messages are recognized by this id.
   */
  MessageID m_id;
  
  /** An message number. */
  int m_messageNum;

  /** Pointer to GameObject who sent this message. If this message was not
   * sent by a GameObject, this should be a null-pointer
   */
  GameObject* m_pSender;

  /** To whom this message is targeted. Use null-pointer if the target is
   * not a GameObject
   */
  GameObject* m_pDestination;

  /** Table of parameters that are delivered along this message. */
  eng2d::ParameterTable m_params;

protected:

  ///
  /// Private members
  /// ===============
  
  /** This member keeps track of the message numbers */
  static int s_messageNumbers;
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: objectmessage.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:08+02  lamminsa
 * Initial revision
 *
 */
 
